// Game state management
class PerimeterStickerGame {
    constructor() {
        this.currentShape = 'rectangle';
        this.currentDimensions = {};
        this.correctPerimeter = 0;
        this.stickersNeeded = 0;
        this.stickersPlaced = 0;
        this.score = 0;
        this.stickerColors = ['#ff6b6b', '#4ecdc4', '#45b7d1', '#96ceb4', '#ffeaa7', '#dda0dd', '#98d8c8', '#f7dc6f'];
        this.stickerSlots = [];
        
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.generateNewShape();
        this.updateDisplay();
    }

    setupEventListeners() {
        // Shape selection buttons
        document.querySelectorAll('.shape-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                this.selectShape(e.target.dataset.shape);
            });
        });

        // Check answer button
        document.getElementById('check-btn').addEventListener('click', () => {
            this.checkPerimeter();
        });

        // New shape button
        document.getElementById('new-shape-btn').addEventListener('click', () => {
            this.generateNewShape();
        });

        // Reset button
        document.getElementById('reset-btn').addEventListener('click', () => {
            this.resetCurrentShape();
        });

        // Continue button in modal
        document.getElementById('continue-btn').addEventListener('click', () => {
            this.hideSuccessModal();
            this.generateNewShape();
        });

        // Enter key for perimeter input
        document.getElementById('perimeter-input').addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                this.checkPerimeter();
            }
        });
    }

    selectShape(shape) {
        // Update active button
        document.querySelectorAll('.shape-btn').forEach(btn => {
            btn.classList.remove('active');
        });
        document.querySelector(`[data-shape="${shape}"]`).classList.add('active');

        // Update active dimensions display
        document.querySelectorAll('.dims-info').forEach(info => {
            info.classList.remove('active');
        });
        document.getElementById(`${shape}-dims`).classList.add('active');

        this.currentShape = shape;
        this.generateNewShape();
    }

    generateNewShape() {
        // Generate random dimensions based on shape type
        switch(this.currentShape) {
            case 'rectangle':
                this.currentDimensions = {
                    length: Math.floor(Math.random() * 8) + 4, // 4-11
                    width: Math.floor(Math.random() * 6) + 3   // 3-8
                };
                this.correctPerimeter = 2 * (this.currentDimensions.length + this.currentDimensions.width);
                break;
            case 'square':
                this.currentDimensions = {
                    side: Math.floor(Math.random() * 6) + 4 // 4-9
                };
                this.correctPerimeter = 4 * this.currentDimensions.side;
                break;
            case 'triangle':
                this.currentDimensions = {
                    a: Math.floor(Math.random() * 5) + 4, // 4-8
                    b: Math.floor(Math.random() * 5) + 4, // 4-8
                    c: Math.floor(Math.random() * 5) + 4  // 4-8
                };
                this.correctPerimeter = this.currentDimensions.a + this.currentDimensions.b + this.currentDimensions.c;
                break;
        }

        this.stickersNeeded = this.correctPerimeter;
        this.stickersPlaced = 0;
        
        this.updateDisplay();
        this.createShape();
        this.createStickerSlots();
        this.createStickers();
        this.updateFormula();
    }

    updateDisplay() {
        // Update dimensions display
        switch(this.currentShape) {
            case 'rectangle':
                document.getElementById('rect-length').textContent = this.currentDimensions.length;
                document.getElementById('rect-width').textContent = this.currentDimensions.width;
                break;
            case 'square':
                document.getElementById('square-side').textContent = this.currentDimensions.side;
                break;
            case 'triangle':
                document.getElementById('tri-a').textContent = this.currentDimensions.a;
                document.getElementById('tri-b').textContent = this.currentDimensions.b;
                document.getElementById('tri-c').textContent = this.currentDimensions.c;
                break;
        }

        // Update sticker counter
        document.getElementById('stickers-needed').textContent = this.stickersNeeded;
        document.getElementById('stickers-placed').textContent = this.stickersPlaced;
        
        // Update score
        document.getElementById('score').textContent = this.score;
        
        // Clear perimeter input
        document.getElementById('perimeter-input').value = '';
    }

    createShape() {
        const container = document.getElementById('shape-container');
        container.innerHTML = '';

        const shape = document.createElement('div');
        shape.className = `shape ${this.currentShape}`;

        switch(this.currentShape) {
            case 'rectangle':
                shape.style.width = `${this.currentDimensions.length * 20}px`;
                shape.style.height = `${this.currentDimensions.width * 20}px`;
                break;
            case 'square':
                const size = this.currentDimensions.side * 20;
                shape.style.width = `${size}px`;
                shape.style.height = `${size}px`;
                break;
            case 'triangle':
                // Create triangle using CSS borders
                const base = Math.max(this.currentDimensions.a, this.currentDimensions.b, this.currentDimensions.c) * 15;
                const height = base * 0.8;
                shape.style.borderLeft = `${base/2}px solid transparent`;
                shape.style.borderRight = `${base/2}px solid transparent`;
                shape.style.borderBottom = `${height}px solid #e1bee7`;
                break;
        }

        container.appendChild(shape);
    }

    createStickerSlots() {
        const slotsContainer = document.getElementById('sticker-slots');
        slotsContainer.innerHTML = '';
        this.stickerSlots = [];

        const shapeElement = document.querySelector('.shape');
        const containerRect = document.getElementById('shape-container').getBoundingClientRect();
        const displayRect = document.querySelector('.shape-display').getBoundingClientRect();

        // Calculate positions for sticker slots around the shape perimeter
        const slots = this.calculateStickerPositions();
        
        slots.forEach((pos, index) => {
            const slot = document.createElement('div');
            slot.className = 'sticker-slot';
            slot.style.left = `${pos.x}px`;
            slot.style.top = `${pos.y}px`;
            slot.dataset.index = index;
            
            // Add drop zone functionality
            this.addDropZoneListeners(slot);
            
            slotsContainer.appendChild(slot);
            this.stickerSlots.push({element: slot, filled: false});
        });
    }

    calculateStickerPositions() {
        const positions = [];
        const centerX = 250; // Center of shape display area
        const centerY = 150;
        
        switch(this.currentShape) {
            case 'rectangle':
                const rectWidth = this.currentDimensions.length * 20;
                const rectHeight = this.currentDimensions.width * 20;
                const startX = centerX - rectWidth/2;
                const startY = centerY - rectHeight/2;
                
                // Top edge
                for(let i = 0; i < this.currentDimensions.length; i++) {
                    positions.push({x: startX + i * 20 + 10, y: startY - 10});
                }
                // Right edge
                for(let i = 0; i < this.currentDimensions.width; i++) {
                    positions.push({x: startX + rectWidth + 10, y: startY + i * 20 + 10});
                }
                // Bottom edge
                for(let i = 0; i < this.currentDimensions.length; i++) {
                    positions.push({x: startX + rectWidth - i * 20 - 10, y: startY + rectHeight + 10});
                }
                // Left edge
                for(let i = 0; i < this.currentDimensions.width; i++) {
                    positions.push({x: startX - 10, y: startY + rectHeight - i * 20 - 10});
                }
                break;
                
            case 'square':
                const size = this.currentDimensions.side * 20;
                const squareStartX = centerX - size/2;
                const squareStartY = centerY - size/2;
                
                // Top edge
                for(let i = 0; i < this.currentDimensions.side; i++) {
                    positions.push({x: squareStartX + i * 20 + 10, y: squareStartY - 10});
                }
                // Right edge
                for(let i = 0; i < this.currentDimensions.side; i++) {
                    positions.push({x: squareStartX + size + 10, y: squareStartY + i * 20 + 10});
                }
                // Bottom edge
                for(let i = 0; i < this.currentDimensions.side; i++) {
                    positions.push({x: squareStartX + size - i * 20 - 10, y: squareStartY + size + 10});
                }
                // Left edge
                for(let i = 0; i < this.currentDimensions.side; i++) {
                    positions.push({x: squareStartX - 10, y: squareStartY + size - i * 20 - 10});
                }
                break;
                
            case 'triangle':
                // Distribute stickers along triangle perimeter
                const {a, b, c} = this.currentDimensions;
                const totalStickers = a + b + c;
                
                // Simple triangle positioning (equilateral approximation)
                const triSize = 80;
                const triHeight = triSize * Math.sqrt(3) / 2;
                
                // Side A (bottom)
                for(let i = 0; i < a; i++) {
                    const t = i / (a - 1);
                    positions.push({
                        x: centerX - triSize/2 + t * triSize,
                        y: centerY + triHeight/3 + 15
                    });
                }
                
                // Side B (left)
                for(let i = 0; i < b; i++) {
                    const t = i / (b - 1);
                    positions.push({
                        x: centerX - triSize/2 + t * triSize/2 - 15,
                        y: centerY + triHeight/3 - t * triHeight
                    });
                }
                
                // Side C (right)
                for(let i = 0; i < c; i++) {
                    const t = i / (c - 1);
                    positions.push({
                        x: centerX + t * triSize/2 + 15,
                        y: centerY - triHeight/3 + t * triHeight
                    });
                }
                break;
        }
        
        return positions;
    }

    createStickers() {
        const container = document.getElementById('stickers-container');
        container.innerHTML = '';

        // Create stickers with random colors
        for(let i = 0; i < this.stickersNeeded + 3; i++) { // Extra stickers for variety
            const sticker = document.createElement('div');
            sticker.className = 'sticker';
            sticker.style.backgroundColor = this.stickerColors[i % this.stickerColors.length];
            sticker.draggable = true;
            
            this.addDragListeners(sticker);
            container.appendChild(sticker);
        }
    }

    addDragListeners(sticker) {
        sticker.addEventListener('dragstart', (e) => {
            sticker.classList.add('dragging');
            e.dataTransfer.effectAllowed = 'move';
            e.dataTransfer.setData('text/html', sticker.outerHTML);
        });

        sticker.addEventListener('dragend', () => {
            sticker.classList.remove('dragging');
        });

        // Touch events for mobile
        sticker.addEventListener('touchstart', (e) => {
            e.preventDefault();
            sticker.classList.add('dragging');
            this.handleTouchMove(e, sticker);
        });
    }

    addDropZoneListeners(slot) {
        slot.addEventListener('dragover', (e) => {
            e.preventDefault();
            slot.classList.add('highlight');
        });

        slot.addEventListener('dragleave', () => {
            slot.classList.remove('highlight');
        });

        slot.addEventListener('drop', (e) => {
            e.preventDefault();
            slot.classList.remove('highlight');
            
            if (!this.stickerSlots[slot.dataset.index].filled) {
                this.placeSticker(slot, e.dataTransfer.getData('text/html'));
            }
        });

        // Touch events
        slot.addEventListener('touchend', (e) => {
            if (this.draggedSticker && !this.stickerSlots[slot.dataset.index].filled) {
                this.placeSticker(slot, this.draggedSticker.outerHTML);
                this.draggedSticker.remove();
                this.draggedSticker = null;
            }
        });
    }

    handleTouchMove(e, sticker) {
        this.draggedSticker = sticker;
        const touch = e.touches[0];
        
        const moveHandler = (e) => {
            e.preventDefault();
            const touch = e.touches[0];
            sticker.style.position = 'fixed';
            sticker.style.left = touch.clientX - 10 + 'px';
            sticker.style.top = touch.clientY - 10 + 'px';
            sticker.style.zIndex = '1000';
        };
        
        const endHandler = () => {
            sticker.classList.remove('dragging');
            sticker.style.position = '';
            sticker.style.left = '';
            sticker.style.top = '';
            sticker.style.zIndex = '';
            document.removeEventListener('touchmove', moveHandler);
            document.removeEventListener('touchend', endHandler);
        };
        
        document.addEventListener('touchmove', moveHandler);
        document.addEventListener('touchend', endHandler);
    }

    placeSticker(slot, stickerHTML) {
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = stickerHTML;
        const newSticker = tempDiv.firstChild;
        
        // Remove drag functionality and resize for slot
        newSticker.draggable = false;
        newSticker.className = 'sticker sticker-place-animation';
        newSticker.style.width = '16px';
        newSticker.style.height = '16px';
        newSticker.style.cursor = 'default';
        
        // Clear the slot and add the sticker
        slot.innerHTML = '';
        slot.appendChild(newSticker);
        slot.classList.add('filled');
        
        // Update game state
        this.stickerSlots[slot.dataset.index].filled = true;
        this.stickersPlaced++;
        
        // Update display
        document.getElementById('stickers-placed').textContent = this.stickersPlaced;
        
        // Check if all stickers are placed
        if (this.stickersPlaced === this.stickersNeeded) {
            this.checkCompletion();
        }
    }

    checkPerimeter() {
        const input = document.getElementById('perimeter-input');
        const userAnswer = parseInt(input.value);
        
        if (userAnswer === this.correctPerimeter) {
            // Correct answer - enable sticker placement
            input.style.borderColor = '#28a745';
            input.style.backgroundColor = '#d4edda';
            
            // Show success feedback
            this.showFeedback('Correct! Now place the stickers around the shape.', 'success');
            
            // Enable sticker slots
            document.querySelectorAll('.sticker-slot').forEach(slot => {
                slot.style.pointerEvents = 'all';
            });
        } else {
            // Wrong answer
            input.style.borderColor = '#dc3545';
            input.style.backgroundColor = '#f8d7da';
            
            this.showFeedback('Try again! Check your calculation.', 'error');
            
            // Shake animation
            input.classList.add('shake');
            setTimeout(() => input.classList.remove('shake'), 500);
        }
    }

    checkCompletion() {
        if (this.stickersPlaced === this.stickersNeeded) {
            this.score += 10;
            document.getElementById('score').textContent = this.score;
            
            // Add success animation to shape
            document.querySelector('.shape').classList.add('success-animation');
            
            // Show success modal
            setTimeout(() => {
                this.showSuccessModal();
            }, 600);
        }
    }

    showSuccessModal() {
        document.getElementById('success-modal').classList.add('show');
    }

    hideSuccessModal() {
        document.getElementById('success-modal').classList.remove('show');
    }

    showFeedback(message, type) {
        // Create temporary feedback element
        const feedback = document.createElement('div');
        feedback.textContent = message;
        feedback.style.cssText = `
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            padding: 10px 20px;
            border-radius: 4px;
            font-size: 14px;
            z-index: 1500;
            ${type === 'success' ? 
                'background: #d4edda; color: #155724; border: 1px solid #c3e6cb;' : 
                'background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;'
            }
        `;
        
        document.body.appendChild(feedback);
        
        setTimeout(() => {
            feedback.remove();
        }, 2000);
    }

    updateFormula() {
        const formulaHint = document.getElementById('formula-hint');
        switch(this.currentShape) {
            case 'rectangle':
                formulaHint.textContent = 'Rectangle: 2 × (length + width)';
                break;
            case 'square':
                formulaHint.textContent = 'Square: 4 × side length';
                break;
            case 'triangle':
                formulaHint.textContent = 'Triangle: side A + side B + side C';
                break;
        }
    }

    resetCurrentShape() {
        // Reset sticker placement
        this.stickersPlaced = 0;
        document.getElementById('stickers-placed').textContent = '0';
        
        // Clear all sticker slots
        this.stickerSlots.forEach(slot => {
            slot.element.innerHTML = '';
            slot.element.classList.remove('filled');
            slot.filled = false;
        });
        
        // Reset input field
        const input = document.getElementById('perimeter-input');
        input.value = '';
        input.style.borderColor = '#dee2e6';
        input.style.backgroundColor = 'white';
        
        // Recreate stickers
        this.createStickers();
    }
}

// Tooltip functionality
function showTooltip(event) {
    const tooltip = document.getElementById('tooltip');
    tooltip.classList.add('show');
}

function hideTooltip() {
    const tooltip = document.getElementById('tooltip');
    tooltip.classList.remove('show');
}

// Initialize game when page loads
document.addEventListener('DOMContentLoaded', () => {
    new PerimeterStickerGame();
});

// Add shake animation CSS
const style = document.createElement('style');
style.textContent = `
    .shake {
        animation: shake 0.5s ease-in-out;
    }
    
    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        25% { transform: translateX(-5px); }
        75% { transform: translateX(5px); }
    }
`;
document.head.appendChild(style);